/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::  Module      :   Variable Cam Phaser Framework API Header File
::  Copyright   :   (C)2005-20099 Woodward
::  Platform(s) :   MPC5xx
::  Limitations :   MPC5xx OS
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_VarCam.h
    \brief The Framework can support variable cam phaser patterns. Reports are issued to the application when
           position feedback points within the pattern are detected. The crank angle position when the feedback
           point is detected is supplied as data when the notification executes. Cam velocity may also be supplied
*/
#ifndef __RESOURCE_VARCAM_H
#define __RESOURCE_VARCAM_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <resource.h>
#include <NativeError.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! The set of supported cam phaser patterns */

/* Name: E_VarCamPattern ClassID:ENUMDEC EnumDflt:"Unknown Cam Phaser Pattern" */
typedef enum
{
/*! A variable cam phaser pattern that is constructed from 6 teeth. 3 of the teeth, which describe 360degCA
    are narrow while the other 3 teeth are wide. Narrow teeth represent one half of the engine cycle and wide
    teeth represent the other half of the cycle. The synchronous edge provides a position feedback point */
    VCAM_3X_PW             = 0, /* EnumTxt:"6 Teeth - Half Wide" */
    VCAM_6_TEETH_HALF_WIDE = 0, /* EnumTxt:"6 Teeth - Half Wide" */
/*! Half moon cam phaser. The lobe describes 360 degCA. Each edge provides position feedback */
    VCAM_HALFMOON          = 1, /* EnumTxt:"Halfmoon" */
/*! A variable cam phaser pattern that is constructed from 8 teeth. 4 of the teeth, which describe 360degCA
    are narrow while the other 4 teeth are wide. Narrow teeth represent one half of the engine cycle and wide
    teeth represent the other half of the cycle. The synchronous edge provides a position feedback point */
    VCAM_4X_PW             = 2, /* EnumTxt:"8 Teeth - Half Wide" */
    VCAM_8_TEETH_HALF_WIDE = 2, /* EnumTxt:"8 Teeth - Half Wide" */
/*! A variable cam phaser pattern that is constructed from 4 teeth. 2 of the teeth, which describe 360degCA
    are narrow while the other 2 teeth are wide. Narrow teeth represent one half of the engine cycle and wide
    teeth represent the other half of the cycle. The synchronous edge provides a position feedback point */
    VCAM_2X_PW             = 3, /* EnumTxt:"4 Teeth - Half Wide" */
    VCAM_4_TEETH_HALF_WIDE = 3, /* EnumTxt:"4 Teeth - Half Wide" */
/*! A Variable cam pattern that is constructed of 6 pulses, with 2 missing (in equally spaced locations)
    and 2 added (close after 2 of the equally spaced ones) ______-_-___-_-_________-____________- */
    VCAM_T6M2P2            = 4, /* EnumTxt:"6 Teeth - 2 Missing, 2 Additional (Plus)" */
    VCAM_NISSANZ           = 4,

/*! A variable cam phaser pattern that is constructed from 4 equally spaced teeth per crank cycle (EVO9). 
    The synchronous edge provides a position feedback point */
    VCAM_EVO9_4X           = 5, /* EnumTxt:"4 Teeth equally spaced per cycle" */

/*! A variable cam phaser pattern that is constructed from 3 equally spaced teeth per crank cycle */
    VCAM_3X                = 6, /* EnumTxt:"3 Teeth equally spaced per cycle" */

/*! A variable cam phaser pattern that is constructed from 4 Minus 1 = 3 teeth per crank cycle */
    VCAM_4M1               = 7, /* EnumTxt:"4 Minus 1 Teeth" */

    /* NOTE: If you add another entry here, also add handler in PFNGETDEGSPERFBEVENT */
    NUM_SUPPORTED_VARCAMS
} E_VarCamPattern;

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid discrete
           output attributes in the S_VarCamResourceAttributes and S_VarCamCreateResourceAttributes data
           structures.

    Each mask definition applies to a particular member of one of the data structures.
    \code
    ...
    S_VarCamResourceAttributes AttribObj;

    AttribObj.uValidAttributesMask = USE_VCAM_CONDITION;
    EncAttribObj.eResourceCond = RES_ENABLED;
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
typedef enum
{
    USE_VCAM_CONDITION = 0x01,          /*!< Selects S_VarCamResourceAttributes::eResourceCond */
    USE_VCAM_POSN_REPORT = 0x02,        /*!< Selects S_VarCamCreateResourceAttributes::pfPosnReport */
    USE_VCAM_SYNC_POLARITY = 0x04,      /*!< Selects S_VarCamCreateResourceAttributes::eSyncPolarity */
    USE_VCAM_PATTERN = 0x08,            /*!< Selects S_VarCamCreateResourceAttributes::ePattern */
    USE_VCAM_DYNAMIC_ON_CREATE = 0x10,  /*!< Selects S_VarCamCreateResourceAttributes::DynamicObj */

/* IF THIS TYPE EXCEEDS 0x80 THEN ALTER THE TYPE OF THE VarCamAttributesMask_U ACCORDINGLY */

} E_VarCamAttributesMask;

/*! Unsigned integer type of sufficient size to hold the attribute masks for a variable cam phaser described by \ref E_VarCamAttributesMask */
typedef uint1 VarCamAttributesMask_U;

/*! \brief Structure that describes the attributes of a variable cam phaser that are dynamic. These attributes can be
           altered as conditions change.

    Attributes are applied using with the SetResourceAttributesBEHAVIOUR_VARCAM() function. Not all the 
    available attributes need to be set by a single call. Each attribute has a mask associated with it.
    Logic-OR this mask into the \ref S_VarCamResourceAttributes::uValidAttributesMask member to enable an
    attribute. */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_VCAM_CONDITION ]
    that are valid for this this instance of the data structure.

    \code
    ...
    S_VarCamResourceAttributes AttribObj;

    AttribObj.uValidAttributesMask = USE_VCAM_CONDITION;
    AttribObj.eResourceCond = RES_ENABLED;
    SetResourceAttributesBEHAVIOUR_VARCAM(RES_CAM, &AttribObj);  \endcode */
    VarCamAttributesMask_U uValidAttributesMask;

/*! Describes the condition of the resource. On creation this attribute is set to \ref RES_DISABLED. Select
    this attribute with the \ref USE_VCAM_CONDITION attribute bit mask. A disabled variable cam phaser shall
    not report position */
    E_ResourceCond eResourceCond;
} S_VarCamResourceAttributes;


/*! \brief Structure describes a variable cam phaser position report */
typedef struct S_VarCamPhaserReport
{
/*! The absolute crank angle position relative firing TDC#1 when the cam phaser feedback position was detected.
    The value is a fixed point integer where 1 degBfTDC#1 == 16 */
    sint2 sPosn;
/*! The rate of change of the cam position between this feedback point and the last observed feedback point.
    The velocity is a fixed point value where 1 degCA/second == 16 */
    sint4 sVelocity;
/*! The amount of time (microseconds) that has passed between this feedback edge and the last observed feedback edge */
    uint4 uTimeBetweenEdges;
/*! A time stamp when the feedback edge was detected. The GetFreeRunningCounter32Bits() function may be used to get
    a related timestamp for the current time. This timestamp will allow the time that has elapsed between when the
    feedback edge occurred and when the information is to be used by the application to be determine. This allows
    the interpolated position of the cam to be calculated using the corresponding velocity value.

    \code    
    uTimeSinceEdge_us = GetFreeRunningCounter32Bits() - S_VarCamPhaserReport::uTimeEdgeOccurred; \endcode */
    uint4 uTimeEdgeOccurred;
} S_VarCamPhaserReport;
/*! \brief Prototype for a variable cam phaser report that is issued (when enabled) when a feedback point of
           the associated variable cam phaser is detected
    
 The \c in_uAppDataToSendOnNotify parameter is the S_VarCamCreateResourceAttributes::pPosnReportUserData data
 that was supplied by the application when this notification event was setup. */
typedef void (*PFNVARCAMPOSNREPORT)(E_ModuleResource, S_VarCamPhaserReport const*, void* pAppDataToSendOnNotify);

/*! \brief Structure describes the attributes of the variable cam phaser behaviour that is to be assigned to the
           module resource during creation.
    
 Use CreateResourceBEHAVIOUR_VARCAM() to create a variable cam phaser. The data structure has some attributes that are optional
 or support defaults. An attribute mask S_VarCamCreateResourceAttributes::uValidAttributesMask is used to identify
 these attributes by logic-ORing those attributes into the mask. */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_VCAM_SYNC_POLARITY, \ref USE_VCAM_POSN_REPORT, \ref USE_VCAM_DYNAMIC_ON_CREATE ]

    \code
    ...
    S_VarCamCreateResourceAttributes CreateAttribObj;

    CreateAttribObj.uValidAttributesMask = USE_VCAM_POSN_REPORT | USE_VCAM_DYNAMIC_ON_CREATE;
    CreateAttribObj.pfPosnReport = MyvarCamPosnReportFunction;
    CreateAttribObj.pPosnReportUserData = NULL;
    CreateAttribObj.DynamicObj.uValidAttributesMask = USE_ENC_CONDITION;
    CreateAttribObj.DynamicObj.eResourceCond = RES_ENABLED;
    CreateResourceBEHAVIOUR_VARCAM(RES_CAM, &CreateAttribObj);  \endcode */
    VarCamAttributesMask_U uValidAttributesMask;
/*! The synchronous edge of the pattern. Describes which edge of a digital signal represents the synchronous
    edge of the tooth. This edge describes the tooth event. The edge is relative to the signal as it enters the
    module. Any inversion that occurs within the module is automatically compensated for by the framework. Use
    the \ref USE_VCAM_SYNC_POLARITY to select this attribute. If unselected then \ref RES_FALL_EDGE_DETECT is
    assumed */
    E_ResourceEdgeSensitivity eSyncPolarity;

/*! Describes the expected pattern that the cam phaser shall use. Select this attribute with the \ref USE_VCAM_PATTERN
    bit mask. Creation shall fail with ERROR_BAD_ATTRIBUTE_DATA if this attribute is not set. */
    E_VarCamPattern ePattern;

/*! Describes the notification function that shall execute when a position feedback point is detected. Select this
    attribute with the \ref USE_VCAM_POSN_REPORT bit mask. Creation shall fail if this attribute is not defined */
    PFNVARCAMPOSNREPORT pfPosnReport;
/*! User defined data, which is normally a pointer, that is supplied when the S_VarCamCreateResourceAttributes::pfPosnReport
    executes. */
    void* pPosnReportUserData;
/*! Dynamic attributes that are applied on creation */
    S_VarCamResourceAttributes DynamicObj;
} S_VarCamCreateResourceAttributes;


#pragma pack()
/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/
NativeError_S CreateResourceBEHAVIOUR_VARCAM(E_ModuleResource, S_VarCamCreateResourceAttributes const*);
NativeError_S SetResourceAttributesBEHAVIOUR_VARCAM(E_ModuleResource, S_VarCamResourceAttributes const*);

#endif /* __RESOURCE_VARCAM_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
